 /**
* \file: EventDispatcher.h
*
* \version: 0.1
*
* \release: $Name:$
*
* Includes the necessary header file to test the Feature Discovery.
*
* \component: BDCL
*
* \author: D. Girnus / ADIT/SW2 / dgirnus@de.adit-jv.com
*          P. Govindaraju / pradeepa.govindaraju@in.bosch.com
*
* \copyright (c) 2016 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#ifndef BDCL_DEVICE_EVENT_DISPATCHER_H_
#define BDCL_DEVICE_EVENT_DISPATCHER_H_


#include <pthread_adit.h>
#include <mutex>

#include <memory>

#include <cstring>
#include <deque>
#include <semaphore.h>

// FeatureDiscovery headers
#include <uspi/FeatureDiscoveryTypes.h>

namespace adit { namespace bdcl {

enum class EventType : int32_t
{
    NONE                = 0,
    FOUND_APPLE_DEVICE  = 1,
    FOUND_AOAP_DEVICE   = 2,
    LOST_APPLE_DEVICE   = 3,
    LOST_AOAP_DEVICE    = 4,
    SWITCHED            = 5,
    CHANGED             = 6,
    EANSTART            = 7, //iAP2 ExternalAccessory Native Session Start on App launch
    EANSTOP             = 8, //iAP2 ExternalAccessory Native Session Stop on App Close or iAP2 Error
    CMDMDEXIT           = 9, //Session related events
    AOAPREADTIMEOUT     = 10, //Session related events
    OTHER               = 11
};


class EventItem
{
public:
    EventItem(std::shared_ptr<adit::uspi::DiscoveredDevice> inDevice, EventType inEvent,std::string readEp, std::string writeEp)
        : mDevice(inDevice),
          mEvent(inEvent),
          mEaReadEp(readEp),      // Not needed for Android devices and will be set to 0;
          mEaWriteEp(writeEp){}   // Not needed for Android devices and will be set to 0;


    virtual ~EventItem() { }

    EventType getEventType() { return mEvent; }
    std::shared_ptr<adit::uspi::DiscoveredDevice> getDevice() { return mDevice; }
    std::string getEaWriteFile(){return mEaWriteEp;}
    std::string getEaReadFile(){return mEaReadEp;}

private:
    std::shared_ptr<adit::uspi::DiscoveredDevice> mDevice;
    EventType mEvent;
    std::string mEaReadEp; //EA Native Read  end point
    std::string mEaWriteEp; //EA native Write end point
};


class EventDispatcher
{
public:
    EventDispatcher();
    virtual ~EventDispatcher();

    bool startEventDispatcher();
    void stopEventDispatcher();

    void queueEvent(const std::shared_ptr<EventItem>& work);

    virtual void onDeviceError(adit::uspi::DiscoveryError inErrorCode) = 0;

protected:
    // to be implemented by user
    virtual void onAppleDeviceFound(std::shared_ptr<adit::uspi::DiscoveredDeviceUsb> inUsbDevice) = 0;
    virtual void onAppleDeviceLost(std::shared_ptr<adit::uspi::DiscoveredDeviceUsb> inUsbDevice) = 0;
    virtual void onAoapDeviceFound(std::shared_ptr<adit::uspi::DiscoveredDeviceUsb> inUsbDevice) = 0;
    virtual void onAoapDeviceLost(std::shared_ptr<adit::uspi::DiscoveredDeviceUsb> inUsbDevice) = 0;
    virtual void onDeviceSwitched(std::shared_ptr<adit::uspi::DiscoveredDeviceUsb> inUsbDevice) = 0;
    virtual void onDeviceChanged(std::shared_ptr<adit::uspi::DiscoveredDeviceUsb> inUsbDevice) = 0;
    virtual void onStartEANative(std::string wFile, std::string rFile) = 0;
    virtual void onStopEANative(std::string wFile, std::string rFile) = 0;

    //Session related events
    virtual void onMDExit(std::string serial) = 0;
    virtual void onAoapReadTimeOutError(std::string serial) = 0;

private:
    int handleEvent(std::shared_ptr<EventItem> inEventItem);
    static void* EventDispatcherThread(void* context);
    pthread_t mEventDispatcherThreadId;
    bool mRunning;

    sem_t mEventAvailable;
    bool mSemCreated;

    std::shared_ptr<EventItem> dequeueEvent();

    std::deque<std::shared_ptr<EventItem> > mEventQueue;
    std::mutex mEventQueueMutex;
};

} } /* namespace adit { namespace uspi { */

#endif /* BDCL_DEVICE_EVENT_DISPATCHER_H_ */
